package mcfall.raytracer;

import java.awt.Dimension;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import mcfall.math.Matrix;
import mcfall.math.Point;
import mcfall.math.Ray;
import mcfall.raytracer.objects.AbstractBoundingBox;
import mcfall.raytracer.objects.AbstractRectangleExtent;
import mcfall.raytracer.objects.AmbientLightSource;
import mcfall.raytracer.objects.GenericCube;
import mcfall.raytracer.objects.HitRecord;
import mcfall.raytracer.objects.ProjectedExtent;
import mcfall.raytracer.objects.RGBIntensity;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
// 
/**
 * The Scene class represents a particular scene, including the objects in the scene and their light sources.  Methods 
 * are provided to add objects to the scene, determine the first object in the scene hit by a given ray, and
 * to obtain a list of objects in the Scene 
 * @author mcfall
 */
public class Scene {
	
	private static Logger logger = Logger.getLogger(Scene.class);
	
	/**  Represents the location and shape of the camera in the scene  **/
	private Camera camera;
	
	/**
	 * Maintains a list of all the objects present in the scene
	 */
	private List<ThreeDimensionalObject> objects;
	
	/**
	 * Maintains information about the lights in the scene
	 */
	private List<LightSource> lights;
	
	
	/**
	 * Maintains a map of materials that are used in multiple places in the scene file.  The key 
	 * for the map is the ID attribute of the material, with the material object itself contained
	 * as the value
	 */
	private Map<String, Material> materials;

	private boolean boundingBoxEnabled;
		
	/**
	 * Creates a new scene object, with no objects in the scene
	 *
	 */
	public Scene () {
		objects = new LinkedList<ThreeDimensionalObject> ();
		lights = new LinkedList<LightSource> ();
		materials = new HashMap<String, Material> ();
	}
	
	/**  Sets the location of the camera in the scene **/
	public void setCamera (Camera camera) {
		this.camera = camera;
	}
	
	/**
	 * Retrieves the camera associated with this scene
	 * @return the camera object in the scene
	 */
	public Camera getCamera() {
		return camera;
	}
	
	/**
	 * Creates a scene from an XML file matching the scene Document Type Definition
	 * @param sceneFile
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws XPathExpressionException
	 * @throws InvocationTargetException 
	 * @throws IllegalAccessException 
	 * @throws InstantiationException 
	 * @throws NoSuchMethodException 
	 * @throws ClassNotFoundException 
	 * @throws IllegalArgumentException 
	 * @throws SecurityException 
	 */
	public Scene (String sceneFile) throws FileNotFoundException, IOException, ParserConfigurationException, SAXException, XPathExpressionException, SecurityException, IllegalArgumentException, ClassNotFoundException, NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException {
		this ();
		XPath xpath = XPathFactory.newInstance().newXPath();
		DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
		Document doc = builder.parse(new File(sceneFile));
		//  Read in the description of the scene; right now nothing being done with these items
		String title = xpath.evaluate ("/Scene/@title", doc);
		String author = xpath.evaluate("/Scene/@author", doc);
		String date = xpath.evaluate("/Scene/@date", doc);
		
		//  Read in the information about the camera
		readCamera (xpath, doc);
		
		//  Read in the material list
		readMaterialList (xpath, doc);
		
		//  Now read in and create the lights
		readLights (xpath, doc);
		
		//  Read and create the objects in the scene
		readObjects (xpath, doc);
	}
	
	/**
	 * This method reads in the list of predefined materials and stores them in a map for later use
	 * @param xpath An XPath object that can be used to traverse the document
	 * @param doc The root node of the document
	 * @throws XPathExpressionException if the xpath expression used is malformed
	 */
	private void readMaterialList(XPath xpath, Document doc) throws XPathExpressionException {
		/*  Retrieve a (possibly non-existent) Materials node  */ 
		Node materialList = (Node) xpath.evaluate("/Scene/Materials", doc, XPathConstants.NODE);
		if (materialList == null) return;
		
		/*
		 * Iterate through each material 
		 */
		NodeList materials = (NodeList) xpath.evaluate("Material", materialList, XPathConstants.NODESET);
		for (int i = 0; i < materials.getLength(); i++) {
			createMaterialFromNode(xpath, materials.item(i)); 
		}
	}

	private void readCamera(XPath xpath, Document doc) throws XPathExpressionException {
		Node cameraNode = (Node) xpath.evaluate ("/Scene/Camera", doc, XPathConstants.NODE);
		double viewAngle = Double.valueOf(xpath.evaluate("@viewAngle", cameraNode));
		Point location = readXYZ (xpath, cameraNode, "Location");
		Point lookAt = readXYZ (xpath, cameraNode, "LookAt");
		
		Node viewPlaneNode = (Node) xpath.evaluate("ViewPlane", cameraNode, XPathConstants.NODE);
		int width = Integer.valueOf(xpath.evaluate("@width", viewPlaneNode));
		int height = Integer.valueOf(xpath.evaluate("@height", viewPlaneNode));
		double distance = Double.valueOf(xpath.evaluate("@distance", viewPlaneNode));
		
		Camera camera = new Camera (location, lookAt, viewAngle, width/height, distance, new Dimension (width, height));
		setCamera (camera);
	}

	private Point readXYZ(XPath xpath, Node parent, String tagName) throws XPathExpressionException {
		Node locationNode = (Node) xpath.evaluate (tagName, parent, XPathConstants.NODE);
		double x = Double.valueOf(xpath.evaluate ("@x", locationNode));
		double y = Double.valueOf(xpath.evaluate ("@y", locationNode));
		double z = Double.valueOf(xpath.evaluate ("@z", locationNode));
		return new Point (x, y, z);
	}
	
	/**
	 * Determines the first object in the scene that is hit by a particular
	 * ray.
	 * 
	 * @param ray A <i>Ray</i> object to cast into the scene
	 * @param objects the objects to test
	 * 
	 * @return A HitRecord object describing the hit that occurred; if the ray
	 * didn't hit any objects, null is returned
	 */
	public HitRecord firstObjectHitBy (Ray ray) {
		HitRecord minimumHitRecord = null;
		for (ThreeDimensionalObject object : objects) {			
			List<HitRecord> hits = object.hitTime(ray);
			if (hits.size() > 0) {
				HitRecord[] hitArray = new HitRecord[hits.size()];
				hitArray = hits.toArray(hitArray);
				Arrays.sort(hitArray);
				//  Find the first non-negative value in the array
				for (int i = 0; i < hitArray.length; i++) {
					if (hitArray[i].hitTime > 0) {
						if (minimumHitRecord == null || hitArray[i].hitTime < minimumHitRecord.hitTime) {
							minimumHitRecord = hitArray[i];
							break;
						}
					}
				}				
			}
		}		
		return minimumHitRecord;
	}
	
	/**
	 * Adds a new object into the scene
	 * @param newObject the object to be placed in the scene
	 */
	public void addObject (ThreeDimensionalObject newObject) {
		objects.add(newObject);
	}

	/**
	 * Adds a new light source into the scene
	 * @param light the light source to add to the scene
	 */
	public void addLight (LightSource light) {
		lights.add(light);
	}
	
	/**
	 * Removes the given light source from the scene
	 * @param light the light source to remove
	 */
	public void removeLight (LightSource light) {
		lights.remove(light);
	}
	
	/**
	 * Returns a list containing all of the objects in the scene.  Changes to this list are <b>not</b>
	 * reflected in the scene
	 * @return a new List containing references to the objects in the scene
	 */
	public List<ThreeDimensionalObject> getObjectList() {
		return new ArrayList<ThreeDimensionalObject> (objects);
	}
	
	/**
	 * Returns a list containing all of the lights in the scene.  Changes to this list are <b>not</b>
	 * reflected in the scene
	 * @return a new List containing references to the lights in the scene
	 */
	public List<LightSource> getLights () {
		return new ArrayList<LightSource> (lights);
	}
	
	private void readObjects(XPath xpath, Document doc) throws XPathExpressionException, ClassNotFoundException, SecurityException, NoSuchMethodException, IllegalArgumentException, InstantiationException, IllegalAccessException, InvocationTargetException {
		//  Get all the children of the ObjectList element
		NodeList objectList = (NodeList) xpath.evaluate("/Scene/ObjectList/*", doc, XPathConstants.NODESET);
		for (int i = 0; i < objectList.getLength(); i++) {
			Node objectNode = objectList.item(i);
			
			String objectType = objectNode.getNodeName();
			ThreeDimensionalObject currentObject = null;
			
			if (objectType.equals ("GenericObject")) {
				String specificType = xpath.evaluate("@type", objectNode);
				String name = xpath.evaluate ("@name", objectNode);
				
				/**  Use reflection to invoke the constructor for the appropriate class  **/
				Class objectClass = Class.forName ("mcfall.raytracer.objects.Generic" + specificType);
				Constructor emptyConstructor = objectClass.getConstructor();
				Constructor nameConstructor = objectClass.getConstructor(java.lang.String.class);
									
				if (name == null || name.equals("")) {
					currentObject = (ThreeDimensionalObject) emptyConstructor.newInstance();
				}
				else {
					currentObject = (ThreeDimensionalObject) nameConstructor.newInstance(name);
				}															
				
				Material material = readMaterial (xpath, objectNode);
				currentObject.setMaterial(material);
				
				String reflectionCoefficient = xpath.evaluate("@reflection", objectNode);
				if (reflectionCoefficient != null && !reflectionCoefficient.equals("")) {
					currentObject.setReflectionCoefficient(Double.valueOf(reflectionCoefficient));
				}
				String refractionIndex = xpath.evaluate("@refraction", objectNode);
				if(refractionIndex != null && !refractionIndex.equals("")) {
					currentObject.setRefractionIndex(Double.valueOf(refractionIndex));
				}
				String transparency = xpath.evaluate("@transparency", objectNode);
				if(transparency != null && !transparency.equals("")) {
					currentObject.setTransparency(Double.valueOf(transparency));
					if(currentObject.getRefractionIndex()==0d)
					{
						currentObject.setRefractionIndex(1.0d); //make the object transparent see RayTracer.java
					}
				}
			}
			
			if (objectType.equals ("Mesh")) {
				continue;
			}
			
			readAndApplyTransform (currentObject, xpath, objectNode);
			addObject(currentObject);
			if(currentObject instanceof ProjectedExtent) {
				((ProjectedExtent)currentObject).setUp(camera);
			}
			
		}
	}

	/**
	 * Reads in the transformations associated with the 3D object from a scene file, and 
	 * appends them to the object's transformation
	 * @param object
	 * @param xpath
	 * @param objectNode
	 * @throws XPathExpressionException 
	 */
	private void readAndApplyTransform(ThreeDimensionalObject object, XPath xpath, Node objectNode) throws XPathExpressionException {
		Matrix identity = Matrix.createIdentityMatrix(4);
		
		NodeList transformList = (NodeList) xpath.evaluate ("Transform/*", objectNode, XPathConstants.NODESET);
		
		for (int i=0; i < transformList.getLength(); i++) {
			Node transformNode = transformList.item(i);
			String transformType = transformNode.getNodeName();
			
			if (transformType.equals("Scale")) {
				double x = 1.0;
				double y = 1.0;
				double z = 1.0;
				
				String xScale = xpath.evaluate("@x", transformNode);
				if (!xScale.equals("")) {
					x = Double.valueOf(xScale);
				}
				
				String yScale = xpath.evaluate("@y", transformNode);
				if (!yScale.equals("")) {
					y = Double.valueOf(yScale);
				}
				
				String zScale = xpath.evaluate("@z", transformNode);
				if (!zScale.equals("")) {
					z = Double.valueOf(zScale);
				}
				
				
				Matrix scaleMatrix = Matrix.createScalingMatrix(x, y, z);
				
				
				object.transform(scaleMatrix);
				if (logger.isDebugEnabled()) {
					logger.debug("Scaling " + object.getName() + " by x= " + x + ", y = " + y + ", " + z);
					logger.debug("Transformation matrix for " + object.getName() + " is now \n" + object.getTransform ());
				}
				
			}
			
			if (transformType.equals("Rotate")) {
				double angle = Double.valueOf (xpath.evaluate ("@angle", transformNode));
				Point direction = readXYZ (xpath, transformNode, "Direction");
				direction.setValueAt(4, 0.0);
				Matrix rotationMatrix = Matrix.createRotationMatrix(angle, direction);
				object.transform(rotationMatrix);
				
				if (logger.isDebugEnabled()) {
					logger.debug ("Rotating " + object.getName() + " by " + angle + " degrees around " + direction);
					logger.debug ("Transformation matrix for " + object.getName() + " is now \n" + object.getTransform());
				}
			}
			
			if (transformType.equals("Translate")) {
				double x = 0.0;
				double y = 0.0;
				double z = 0.0;
				
				String xTranslate = xpath.evaluate("@x", transformNode);
				if (!xTranslate.equals("")) {
					x = Double.valueOf(xTranslate);
				}
				
				String yTranslate = xpath.evaluate("@y", transformNode);
				if (!yTranslate.equals("")) {
					y = Double.valueOf(yTranslate);
				}
				
				String zTranslate = xpath.evaluate("@z", transformNode);
				if (!zTranslate.equals("")) {
					z = Double.valueOf(zTranslate);
				}
				
				object.transform(Matrix.createTranslationMatrix(x, y, z));
				if (logger.isDebugEnabled()) {
					logger.debug("Translating " + object.getName() + " by x= " + x + ", y = " + y + ", " + z);
					logger.debug ("Transformation matrix for " + object.getName() + " is now \n" + object.getTransform());
				}
			}
		}
	}

	private void readLights(XPath xpath, Document doc) throws XPathExpressionException {
		//  Get all the children of the LightSources node
		NodeList sourcesNode = (NodeList) xpath.evaluate ("/Scene/LightSources/*", doc, XPathConstants.NODESET);
		for (int i=0; i < sourcesNode.getLength(); i++) {
			Node lightNode = sourcesNode.item(i);
			
			String lightType = lightNode.getNodeName();
			RGBIntensity color = readColor (xpath, lightNode);
									
			if (lightType.equals("Ambient")) {
				addLight(new AmbientLightSource (color.getRed(), color.getGreen(), color.getBlue()));
				continue;
			}
			
			if (lightType.equals("Directional")) {
				double x = Double.valueOf(xpath.evaluate("Location/@x", lightNode));
				double y = Double.valueOf(xpath.evaluate("Location/@y", lightNode));
				double z = Double.valueOf(xpath.evaluate("Location/@z", lightNode));
				addLight(new LightSource (color.getRed(), color.getGreen(), color.getBlue(), x, y, z));
				continue;
			}
			if (lightType.equals("Spotlight")) {
				logger.info("Spotlights not currently implmented");
			}
			logger.error("Invalid light type specified: " + lightType);
		}
		
	}

	private RGBIntensity readColor(XPath xpath, Node lightNode) throws NumberFormatException, XPathExpressionException {
		double red = Double.valueOf(xpath.evaluate ("Color/@red", lightNode));
		double green = Double.valueOf(xpath.evaluate ("Color/@green", lightNode));
		double blue= Double.valueOf(xpath.evaluate ("Color/@blue", lightNode));
		
		return new RGBIntensity (red, green, blue);
	}

	private Material readMaterial (XPath xpath, Node parentNode) throws XPathExpressionException {
		Node materialNode = (Node) xpath.evaluate ("Material", parentNode, XPathConstants.NODE);
	
		/*
		 * Check to see if this material references a pre-defined element in the Materials list.
		 * If so, return that material
		 */
		String idref = xpath.evaluate("@ref", materialNode);
		if (idref != null && !idref.equals("")) {
			Material referencedMaterial = materials.get(idref);
			if (referencedMaterial == null) {
				throw new RuntimeException ("Non-existent material with id " + idref + " referenced");
			}
			return materials.get(idref);
		}
		
		return createMaterialFromNode(xpath, materialNode);
	}

	private Material createMaterialFromNode(XPath xpath, Node materialNode) throws XPathExpressionException {
		Node ambientNode = (Node) xpath.evaluate ("Ambient", materialNode, XPathConstants.NODE);
		Node diffuseNode = (Node) xpath.evaluate ("Diffuse", materialNode, XPathConstants.NODE);
		Node specularNode = (Node) xpath.evaluate ("Specular", materialNode, XPathConstants.NODE);
		
		RGBIntensity ambient = readColor (xpath, ambientNode);
		RGBIntensity diffuse = readColor (xpath, diffuseNode);
		RGBIntensity specular = readColor (xpath, specularNode);
		
		Material newMaterial = new Material (ambient, diffuse, specular);
		String id = xpath.evaluate("@id", materialNode);
		if (id != null && !id.equals("")) {
			materials.put(id, newMaterial);
		}
		return newMaterial;
	}

	public void setBoundingBoxEnabled(boolean boundingBoxEnabled) {
		this.boundingBoxEnabled=boundingBoxEnabled;
	}

	public boolean isBoundingBoxEnabled() {
		return this.boundingBoxEnabled;
	}

	
}
